import { Injectable, inject } from '@angular/core';

import { Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetShippingsAction,
  CreateShippingAction,
  EditShippingAction,
  UpdateShippingAction,
  DeleteShippingAction,
  CreateShippingRuleAction,
  UpdateShippingRuleAction,
  DeleteShippingRuleAction,
} from '../action/shipping.action';
import { IShipping } from '../interface/shipping.interface';
import { NotificationService } from '../services/notification.service';
import { ShippingService } from '../services/shipping.service';

export class ShippingStateModel {
  shipping = {
    data: [] as IShipping[],
  };
  selectedShipping: IShipping | null;
}

@State<ShippingStateModel>({
  name: 'shipping',
  defaults: {
    shipping: {
      data: [],
    },
    selectedShipping: null,
  },
})
@Injectable()
export class ShippingState {
  private notificationService = inject(NotificationService);
  private shippingService = inject(ShippingService);

  @Selector()
  static shipping(state: ShippingStateModel) {
    return state.shipping;
  }

  @Selector()
  static selectedShipping(state: ShippingStateModel) {
    return state.selectedShipping;
  }

  @Action(GetShippingsAction)
  getShippings(ctx: StateContext<ShippingStateModel>) {
    return this.shippingService.getShippings().pipe(
      tap({
        next: result => {
          ctx.patchState({
            shipping: {
              data: result,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreateShippingAction)
  create(ctx: StateContext<ShippingStateModel>, action: CreateShippingAction) {
    return this.shippingService.createShipping(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            shipping: {
              data: [...state.shipping.data, ...result],
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Shipping Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditShippingAction)
  edit(ctx: StateContext<ShippingStateModel>, { id }: EditShippingAction) {
    return this.shippingService.editShipping(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedShipping: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateShippingAction)
  update(ctx: StateContext<ShippingStateModel>, { payload, id }: UpdateShippingAction) {
    return this.shippingService.updateShipping(id, payload).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const data = [...state.shipping.data];
            const index = data.findIndex(shipping => shipping.id === id);
            data[index] = result;

            ctx.patchState({
              ...state,
              shipping: {
                data: data,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Shipping Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteShippingAction)
  delete(ctx: StateContext<ShippingStateModel>, { id }: DeleteShippingAction) {
    return this.shippingService.deleteShipping(id).pipe(
      tap({
        next: () => {
          const state = ctx.getState();
          let data = state.shipping.data.filter(value => value.id !== id);
          ctx.patchState({
            ...state,
            shipping: {
              data: data,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Shipping Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreateShippingRuleAction)
  createRule(ctx: StateContext<ShippingStateModel>, action: CreateShippingRuleAction) {
    return this.shippingService.createShippingRule(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          if (state.selectedShipping) {
            const rule = [...state.selectedShipping.shipping_rules, result];
            state.selectedShipping.shipping_rules = rule;
            ctx.patchState({
              ...state,
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Shipping Rule Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateShippingRuleAction)
  updateRule(ctx: StateContext<ShippingStateModel>, { payload, id }: UpdateShippingRuleAction) {
    return this.shippingService.updateShippingRule(id, payload).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            if (state.selectedShipping) {
              const rule = [...state.selectedShipping.shipping_rules];
              const index = rule.findIndex(rule => rule.id === id);
              rule[index] = result;
              state.selectedShipping.shipping_rules = rule;
              ctx.patchState({
                ...state,
              });
            }
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Shipping Rule Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteShippingRuleAction)
  deleteRule(ctx: StateContext<ShippingStateModel>, { id }: DeleteShippingRuleAction) {
    return this.shippingService.deleteShippingRule(id).pipe(
      tap({
        next: () => {
          const state = ctx.getState();
          if (state.selectedShipping) {
            const rule = state.selectedShipping.shipping_rules.filter(value => value.id !== id);
            state.selectedShipping.shipping_rules = rule;
            ctx.patchState({
              ...state,
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Shipping Rule Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
