import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import { UpdateBadgeValueAction } from '../action/sidebar.action';
import {
  GetStoresAction,
  CreateStoreAction,
  EditStoreAction,
  UpdateStoreAction,
  UpdateStoreStatusAction,
  DeleteStoreAction,
  DeleteAllStoreAction,
  ApproveStoreStatusAction,
} from '../action/store.action';
import { IStores } from '../interface/store.interface';
import { NotificationService } from '../services/notification.service';
import { StoreService } from '../services/store.service';

export class StoreStateModel {
  store = {
    data: [] as IStores[],
    total: 0,
  };
  selectedStore: IStores | null;
}

@State<StoreStateModel>({
  name: 'store',
  defaults: {
    store: {
      data: [],
      total: 0,
    },
    selectedStore: null,
  },
})
@Injectable()
export class StoreState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private storeService = inject(StoreService);

  @Selector()
  static store(state: StoreStateModel) {
    return state.store;
  }

  @Selector()
  static stores(state: StoreStateModel) {
    return state.store.data.map(store => {
      return { label: store?.store_name, value: store?.id };
    });
  }

  @Selector()
  static selectedStore(state: StoreStateModel) {
    return state.selectedStore;
  }

  @Action(GetStoresAction)
  getStores(ctx: StateContext<StoreStateModel>, action: GetStoresAction) {
    return this.storeService.getStores(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            store: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreateStoreAction)
  create(ctx: StateContext<StoreStateModel>, action: CreateStoreAction) {
    return this.storeService.createStore(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            store: {
              data: [...state.store.data, result],
              total: state?.store.total + 1,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Store Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditStoreAction)
  edit(ctx: StateContext<StoreStateModel>, { id }: EditStoreAction) {
    return this.storeService.editStore(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedStore: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateStoreAction)
  update(ctx: StateContext<StoreStateModel>, { payload, id }: UpdateStoreAction) {
    return this.storeService.updateStore(id, payload).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const stores = [...state.store.data];
            const index = stores.findIndex(store => store.id === id);
            stores[index] = result;

            ctx.patchState({
              ...state,
              store: {
                data: stores,
                total: state.store.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Store Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateStoreStatusAction)
  updateStatus(ctx: StateContext<StoreStateModel>, { id, status }: UpdateStoreStatusAction) {
    return this.storeService.updateStoreStatus(id, status).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const stores = [...state.store.data];
            const index = stores.findIndex(store => store.id === id);
            stores[index] = result;

            ctx.patchState({
              ...state,
              store: {
                data: stores,
                total: state.store.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Store Status Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(ApproveStoreStatusAction)
  approveStatus(ctx: StateContext<StoreStateModel>, { id, status }: ApproveStoreStatusAction) {
    return this.storeService.approveStoreStatus(id, status).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const stores = [...state.store.data];
            const index = stores.findIndex(store => store.id === id);
            stores[index] = result;

            ctx.patchState({
              ...state,
              store: {
                data: stores,
                total: state.store.total,
              },
            });
            this.store.dispatch(
              new UpdateBadgeValueAction('/store', result?.total_in_approved_stores),
            );
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Store Status Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteStoreAction)
  delete(ctx: StateContext<StoreStateModel>, { id }: DeleteStoreAction) {
    return this.storeService.deleteStore(id).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetStoresAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Store Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteAllStoreAction)
  deleteAll(ctx: StateContext<StoreStateModel>, { ids }: DeleteAllStoreAction) {
    return this.storeService.deleteAllStore(ids).pipe(
      tap({
        next: () => {
          const state = ctx.getState();
          let store = state.store.data.filter(value => !ids.includes(value.id));
          this.store.dispatch(new GetStoresAction({ page: 1, paginate: 15 }));
          ctx.patchState({
            ...state,
            store: {
              data: store,
              total: state.store.total - ids.length,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Stores Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
