import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetTagsAction,
  CreateTagAction,
  EditTagAction,
  UpdateTagAction,
  UpdateTagStatusAction,
  DeleteTagAction,
  DeleteAllTagAction,
  ExportTagAction,
  ImportTagAction,
} from '../action/tag.action';
import { ITag } from '../interface/tag.interface';
import { NotificationService } from '../services/notification.service';
import { TagService } from '../services/tag.service';

export class TagStateModel {
  tag = {
    data: [] as ITag[],
    total: 0,
  };
  selectedTag: ITag | null;
}

@State<TagStateModel>({
  name: 'tag',
  defaults: {
    tag: {
      data: [],
      total: 0,
    },
    selectedTag: null,
  },
})
@Injectable()
export class TagState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private tagService = inject(TagService);

  @Selector()
  static tag(state: TagStateModel) {
    return state.tag;
  }

  @Selector()
  static selectedTag(state: TagStateModel) {
    return state.selectedTag;
  }

  @Action(GetTagsAction)
  getTags(ctx: StateContext<TagStateModel>, action: GetTagsAction) {
    return this.tagService.getTags(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            tag: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreateTagAction)
  create(ctx: StateContext<TagStateModel>, action: CreateTagAction) {
    return this.tagService.createTag(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            tag: {
              data: [...state.tag.data, result],
              total: state?.tag.total + 1,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Tag Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditTagAction)
  edit(ctx: StateContext<TagStateModel>, { id }: EditTagAction) {
    return this.tagService.editTag(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedTag: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateTagAction)
  update(ctx: StateContext<TagStateModel>, { payload, id }: UpdateTagAction) {
    return this.tagService.updateTag(id, payload).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const tags = [...state.tag.data];
            const index = tags.findIndex(tag => tag.id === id);
            tags[index] = result;

            ctx.patchState({
              ...state,
              tag: {
                data: tags,
                total: state.tag.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Tag Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateTagStatusAction)
  updateStatus(ctx: StateContext<TagStateModel>, { id, status }: UpdateTagStatusAction) {
    return this.tagService.updateTagStatus(id, status).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const tags = [...state.tag.data];
            const index = tags.findIndex(tag => tag.id === id);
            tags[index] = result;

            ctx.patchState({
              ...state,
              tag: {
                data: tags,
                total: state.tag.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Tag Status Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteTagAction)
  delete(_ctx: StateContext<TagStateModel>, { id }: DeleteTagAction) {
    return this.tagService.deleteTag(id).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetTagsAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Tag Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteAllTagAction)
  deleteAll(_ctx: StateContext<TagStateModel>, { ids }: DeleteAllTagAction) {
    return this.tagService.deleteAllTag(ids).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetTagsAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Tag Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(ImportTagAction)
  import(_ctx: StateContext<TagStateModel>, action: ImportTagAction) {
    return this.tagService.importTag(action.payload).pipe(
      tap({
        next: _result => {
          this.store.dispatch(new GetTagsAction({ page: 1, paginate: 15 }));
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(ExportTagAction)
  export(_ctx: StateContext<TagStateModel>, _action: ExportTagAction) {
    return this.tagService.exportTag().pipe(
      tap({
        next: result => {
          const blob = new Blob([result], { type: 'text/csv' });
          const url = window.URL.createObjectURL(blob);
          const link = document.createElement('a');
          link.href = url;
          link.download = 'tag.csv';
          link.click();
          window.URL.revokeObjectURL(url);
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
