import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetTaxesAction,
  CreateTaxAction,
  EditTaxAction,
  UpdateTaxAction,
  UpdateTaxStatusAction,
  DeleteTaxAction,
  DeleteAllTaxAction,
} from '../action/tax.action';
import { ITax } from '../interface/tax.interface';
import { NotificationService } from '../services/notification.service';
import { TaxService } from '../services/tax.service';

export class TaxStateModel {
  tax = {
    data: [] as ITax[],
    total: 0,
  };
  selectedTax: ITax | null;
}

@State<TaxStateModel>({
  name: 'tax',
  defaults: {
    tax: {
      data: [],
      total: 0,
    },
    selectedTax: null,
  },
})
@Injectable()
export class TaxState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private taxService = inject(TaxService);

  @Selector()
  static tax(state: TaxStateModel) {
    return state.tax;
  }

  @Selector()
  static taxes(state: TaxStateModel) {
    return state.tax.data.map((tax: ITax) => {
      return { label: tax?.name, value: tax?.id };
    });
  }

  @Selector()
  static selectedTax(state: TaxStateModel) {
    return state.selectedTax;
  }

  @Action(GetTaxesAction)
  getTaxes(ctx: StateContext<TaxStateModel>, action: GetTaxesAction) {
    return this.taxService.getTaxes(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            tax: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreateTaxAction)
  create(ctx: StateContext<TaxStateModel>, action: CreateTaxAction) {
    return this.taxService.createTax(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            tax: {
              data: [...state.tax.data, result],
              total: state?.tax.total + 1,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Tax Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditTaxAction)
  edit(ctx: StateContext<TaxStateModel>, { id }: EditTaxAction) {
    return this.taxService.editTax(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedTax: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateTaxAction)
  update(ctx: StateContext<TaxStateModel>, { payload, id }: UpdateTaxAction) {
    return this.taxService.updateTax(id, payload).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const taxes = [...state.tax.data];
            const index = taxes.findIndex(tax => tax.id === id);
            taxes[index] = result;

            ctx.patchState({
              ...state,
              tax: {
                data: taxes,
                total: state.tax.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Tax Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateTaxStatusAction)
  updateStatus(ctx: StateContext<TaxStateModel>, { id, status }: UpdateTaxStatusAction) {
    return this.taxService.updateTaxStatus(id, status).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const taxes = [...state.tax.data];
            const index = taxes.findIndex(tax => tax.id === id);
            taxes[index] = result;

            ctx.patchState({
              ...state,
              tax: {
                data: taxes,
                total: state.tax.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Tax Status Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteTaxAction)
  delete(ctx: StateContext<TaxStateModel>, { id }: DeleteTaxAction) {
    return this.taxService.deleteTax(id).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetTaxesAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Tax Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteAllTaxAction)
  deleteAll(ctx: StateContext<TaxStateModel>, { ids }: DeleteAllTaxAction) {
    return this.taxService.deleteAllTax(ids).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetTaxesAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Tax Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
