import { Injectable, inject } from '@angular/core';

import { Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import { GetThemeOptionAction, UpdateThemeOptionAction } from '../action/theme-option.action';
import { IOption, IThemeOption } from '../interface/theme-option.interface';
import { NotificationService } from '../services/notification.service';
import { ThemeOptionService } from '../services/theme-option.service';

export class ThemeOptionStateModel {
  theme_option: IOption | null;
}

@State<ThemeOptionStateModel>({
  name: 'theme_option',
  defaults: {
    theme_option: null,
  },
})
@Injectable()
export class ThemeOptionState {
  private themeOptionService = inject(ThemeOptionService);
  notificationService = inject(NotificationService);

  @Selector()
  static themeOptions(state: ThemeOptionStateModel) {
    return state.theme_option;
  }

  @Action(GetThemeOptionAction)
  getThemeOptions(ctx: StateContext<ThemeOptionStateModel>) {
    return this.themeOptionService.getThemeOption().pipe(
      tap({
        next: (result: IThemeOption) => {
          ctx.patchState({
            theme_option: result.options,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateThemeOptionAction)
  UpdateThemeOptionAction(
    ctx: StateContext<ThemeOptionStateModel>,
    action: UpdateThemeOptionAction,
  ) {
    return this.themeOptionService.updateThemeOption(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            theme_option: result?.options,
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Theme Option Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
