import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetUsersAction,
  CreateUserAction,
  EditUserAction,
  UpdateUserAction,
  UpdateUserStatusAction,
  DeleteUserAction,
  DeleteAllUserAction,
  CreateUserAddressAction,
  ImportUserAction,
  ExportUserAction,
} from '../action/user.action';
import { IUser } from '../interface/user.interface';
import { NotificationService } from '../services/notification.service';
import { UserService } from '../services/user.service';

export class UserStateModel {
  user = {
    data: [] as IUser[],
    total: 0,
  };
  selectedUser: IUser | null;
}

@State<UserStateModel>({
  name: 'user',
  defaults: {
    user: {
      data: [],
      total: 0,
    },
    selectedUser: null,
  },
})
@Injectable()
export class UserState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private userService = inject(UserService);

  @Selector()
  static user(state: UserStateModel) {
    return state.user;
  }

  @Selector()
  static users(state: UserStateModel) {
    return state.user.data.map(user => {
      return { label: user?.name, value: user?.id };
    });
  }

  @Selector()
  static selectedUser(state: UserStateModel) {
    return state.selectedUser;
  }

  @Action(GetUsersAction)
  getUsers(ctx: StateContext<UserStateModel>, action: GetUsersAction) {
    return this.userService.getUsers(action?.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            user: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreateUserAction)
  create(ctx: StateContext<UserStateModel>, action: CreateUserAction) {
    return this.userService.createUser(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            user: {
              data: [...state.user.data, result],
              total: state?.user.total + 1,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('User Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditUserAction)
  edit(ctx: StateContext<UserStateModel>, { id }: EditUserAction) {
    return this.userService.editUser(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedUser: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateUserAction)
  update(ctx: StateContext<UserStateModel>, { payload, id }: UpdateUserAction) {
    return this.userService.updateUser(id, payload).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const users = [...state.user.data];
            const index = users.findIndex(user => user.id === id);
            users[index] = result;

            ctx.patchState({
              ...state,
              user: {
                data: users,
                total: state.user.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('User Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateUserStatusAction)
  updateStatus(ctx: StateContext<UserStateModel>, { id, status }: UpdateUserStatusAction) {
    return this.userService.updateUserStatus(id, status).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const users = [...state.user.data];
            const index = users.findIndex(user => user.id == id);
            users[index] = result;

            ctx.patchState({
              ...state,
              user: {
                data: users,
                total: state.user.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('User Status Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteUserAction)
  delete(_ctx: StateContext<UserStateModel>, { id }: DeleteUserAction) {
    return this.userService.deleteUser(id).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetUsersAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('User Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteAllUserAction)
  deleteAll(_ctx: StateContext<UserStateModel>, { ids }: DeleteAllUserAction) {
    return this.userService.deleteAllUser(ids).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetUsersAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Users Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(ImportUserAction)
  import(_ctx: StateContext<UserStateModel>, action: ImportUserAction) {
    return this.userService.importUser(action.payload).pipe(
      tap({
        next: _result => {
          this.store.dispatch(new GetUsersAction({ page: 1, paginate: 15 }));
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(ExportUserAction)
  export(_ctx: StateContext<UserStateModel>, _action: ExportUserAction) {
    return this.userService.exportUser().pipe(
      tap({
        next: result => {
          const blob = new Blob([result], { type: 'text/csv' });
          const url = window.URL.createObjectURL(blob);
          const link = document.createElement('a');
          link.href = url;
          link.download = 'users.csv';
          link.click();
          window.URL.revokeObjectURL(url);
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreateUserAddressAction)
  createUserAddress(_ctx: StateContext<UserStateModel>, action: CreateUserAddressAction) {
    return this.userService.createUserAddress(action.payload).pipe(
      tap({
        complete: () => {
          this.notificationService.showSuccess('Address Added Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
