import { Injectable, inject } from '@angular/core';

import { Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetUserTransactionAction,
  CreditWalletAction,
  DebitWalletAction,
} from '../action/wallet.action';
import { ITransactionsData } from '../interface/wallet.interface';
import { NotificationService } from '../services/notification.service';
import { WalletService } from '../services/wallet.service';

export class WalletStateModel {
  wallet = {
    consumer_id: null as number | null,
    balance: 0 as number,
    transactions: {
      data: [] as ITransactionsData[],
      total: 0,
    },
  };
}

@State<WalletStateModel>({
  name: 'wallet',
  defaults: {
    wallet: {
      consumer_id: null,
      balance: 0 as number,
      transactions: {
        data: [],
        total: 0,
      },
    },
  },
})
@Injectable()
export class WalletState {
  private notificationService = inject(NotificationService);
  private walletService = inject(WalletService);

  @Selector()
  static wallet(state: WalletStateModel) {
    return state.wallet;
  }

  @Action(GetUserTransactionAction)
  getUserTransations(ctx: StateContext<WalletStateModel>, { payload }: GetUserTransactionAction) {
    return this.walletService.getUserTransaction(payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            wallet: {
              consumer_id: result?.consumer_id,
              balance: result?.balance,
              transactions: {
                data: result?.transactions?.data,
                total: result?.transactions?.total
                  ? result?.transactions?.total
                  : result?.transactions?.data?.length,
              },
            },
          });
        },
        error: err => {
          ctx.patchState({
            wallet: {
              consumer_id: null,
              balance: 0,
              transactions: {
                data: [],
                total: 0,
              },
            },
          });
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreditWalletAction)
  credit(ctx: StateContext<WalletStateModel>, action: CreditWalletAction) {
    return this.walletService.credit(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            wallet: {
              consumer_id: result?.consumer_id,
              balance: result?.balance,
              transactions: {
                data: result?.transactions?.data,
                total: result?.transactions?.total
                  ? result?.transactions?.total
                  : result?.transactions?.data?.length,
              },
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Balance Credited Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DebitWalletAction)
  debit(ctx: StateContext<WalletStateModel>, action: DebitWalletAction) {
    return this.walletService.debit(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            wallet: {
              consumer_id: result?.consumer_id,
              balance: result?.balance,
              transactions: {
                data: result?.transactions?.data,
                total: result?.transactions?.total
                  ? result?.transactions?.total
                  : result?.transactions?.data?.length,
              },
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Balance Debited Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
