import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import { UpdateBadgeValueAction } from '../action/sidebar.action';
import {
  GetWithdrawRequestAction,
  UpdateWithdrawStatusAction,
  WithdrawRequestAction,
} from '../action/withdrawal.action';
import { IWithdrawal } from '../interface/withdrawal.interface';
import { NotificationService } from '../services/notification.service';
import { WithdrawalService } from '../services/withdrawal.service';

export class WithdrawalStateModel {
  withdrawal = {
    data: [] as IWithdrawal[],
    total: 0,
  };
}

@State<WithdrawalStateModel>({
  name: 'withdrawal',
  defaults: {
    withdrawal: {
      data: [],
      total: 0,
    },
  },
})
@Injectable()
export class WithdrawalState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private withdrawalService = inject(WithdrawalService);

  @Selector()
  static withdrawal(state: WithdrawalStateModel) {
    return state.withdrawal;
  }

  @Action(GetWithdrawRequestAction)
  getWithdrawal(ctx: StateContext<WithdrawalStateModel>, action: GetWithdrawRequestAction) {
    return this.withdrawalService.getWithdrawRequest(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            withdrawal: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(WithdrawRequestAction)
  createRequest(ctx: StateContext<WithdrawalStateModel>, action: WithdrawRequestAction) {
    return this.withdrawalService.withdrawRequest(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            withdrawal: {
              data: [...state.withdrawal.data, result],
              total: state?.withdrawal.total + 1,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Your Request Send Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateWithdrawStatusAction)
  updateWithdrawStatus(
    ctx: StateContext<WithdrawalStateModel>,
    { id, status }: UpdateWithdrawStatusAction,
  ) {
    return this.withdrawalService.updateWithdrawStatus(id, status).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const withdrawals = [...state.withdrawal.data];
            const index = withdrawals.findIndex(withdrawal => withdrawal.id === id);
            withdrawals[index] = result;

            ctx.patchState({
              ...state,
              withdrawal: {
                data: withdrawals,
                total: state.withdrawal.total,
              },
            });
            this.store.dispatch(
              new UpdateBadgeValueAction('/withdrawal', result?.total_pending_withdraw_requests),
            );
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Withdraw Status Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
