import { Injectable, inject } from '@angular/core';

import { Action, Selector, State, StateContext, Store } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  CreateZoneAction,
  DeleteAllZoneAction,
  DeleteZoneAction,
  EditZoneAction,
  GetZonesAction,
  UpdateZoneAction,
} from '../action/zone.action';
import { IZone } from '../interface/zone.interface';
import { NotificationService } from '../services/notification.service';
import { ZoneService } from '../services/zone.service';

export class ZoneStateModel {
  zone = {
    data: [] as IZone[],
    total: 0,
  };
  selectedZone: IZone | null;
}

@State<ZoneStateModel>({
  name: 'zone',
  defaults: {
    zone: {
      data: [],
      total: 0,
    },
    selectedZone: null,
  },
})
@Injectable()
export class ZoneState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private zoneService = inject(ZoneService);

  @Selector()
  static zone(state: ZoneStateModel) {
    return state.zone;
  }

  @Selector()
  static zones(state: ZoneStateModel) {
    return state.zone.data.map(res => {
      return { label: res?.name, value: res?.id };
    });
  }

  @Selector()
  static selectedZone(state: ZoneStateModel) {
    return state.selectedZone;
  }

  @Action(GetZonesAction)
  getZones(ctx: StateContext<ZoneStateModel>, action: GetZonesAction) {
    return this.zoneService.getZones(action?.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            zone: {
              data: result.data,
              total: result?.total ? result?.total : result.data.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreateZoneAction)
  create(ctx: StateContext<ZoneStateModel>, action: CreateZoneAction) {
    return this.zoneService.createZone(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            zone: {
              data: [...state.zone.data, result],
              total: state?.zone.total + 1,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Zone Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditZoneAction)
  edit(ctx: StateContext<ZoneStateModel>, { id }: EditZoneAction) {
    return this.zoneService.editZone(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedZone: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateZoneAction)
  update(ctx: StateContext<ZoneStateModel>, { payload, id }: UpdateZoneAction) {
    return this.zoneService.updateZone(id, payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          const zones = [...state.zone.data];
          const index = zones.findIndex(zone => zone.id == id);
          zones[index] = result;

          ctx.patchState({
            ...state,
            zone: {
              data: zones,
              total: state.zone.total - 1,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Zone Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteZoneAction)
  delete(ctx: StateContext<ZoneStateModel>, { id }: DeleteZoneAction) {
    return this.zoneService.deleteZone(id).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetZonesAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Zone Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteAllZoneAction)
  deleteAll(ctx: StateContext<ZoneStateModel>, { ids }: DeleteAllZoneAction) {
    return this.zoneService.deleteAllZone(ids).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetZonesAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Zone Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
