import { CommonModule } from '@angular/common';
import { Component, inject } from '@angular/core';
import {
  FormBuilder,
  FormControl,
  FormGroup,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { Router, RouterModule } from '@angular/router';

import { TranslateModule } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Select2Module } from 'ng-select2-component';
import { Observable } from 'rxjs';

import { RegisterAction } from '../../../shared/action/auth.action';
import { Breadcrumb } from '../../../shared/components/widgets/breadcrumb/breadcrumb';
import { Button } from '../../../shared/components/widgets/button/button';
import * as data from '../../../shared/data/country-code';
import { IBreadcrumb } from '../../../shared/interface/breadcrumb';
import { IValues } from '../../../shared/interface/setting.interface';
import { IOption } from '../../../shared/interface/theme-option.interface';
import { SettingState } from '../../../shared/state/setting.state';
import { ThemeOptionState } from '../../../shared/state/theme-option.state';
import { CustomValidators } from '../../../shared/validator/password-match';

@Component({
  selector: 'app-register',
  imports: [
    CommonModule,
    TranslateModule,
    ReactiveFormsModule,
    Select2Module,
    RouterModule,
    Breadcrumb,
    Button,
  ],
  templateUrl: './register.html',
  styleUrl: './register.scss',
})
export class Register {
  private router = inject(Router);
  private store = inject(Store);
  private formBuilder = inject(FormBuilder);

  setting$: Observable<IValues> = inject(Store).select(SettingState.setting) as Observable<IValues>;
  themeOption$: Observable<IOption> = inject(Store).select(
    ThemeOptionState.themeOptions,
  ) as Observable<IOption>;

  public form: FormGroup;
  public breadcrumb: IBreadcrumb = {
    title: 'Sign In',
    items: [{ label: 'Sign In', active: true }],
  };
  public codes = data.countryCodes;
  public tnc = new FormControl(false, [Validators.requiredTrue]);

  public reCaptcha: boolean = true;

  constructor() {
    this.form = this.formBuilder.group(
      {
        name: new FormControl('', [Validators.required]),
        email: new FormControl('', [Validators.required, Validators.email]),
        phone: new FormControl('', [Validators.required, Validators.pattern(/^[0-9]*$/)]),
        country_code: new FormControl('91', [Validators.required]),
        password: new FormControl('', [Validators.required]),
        password_confirmation: new FormControl('', [Validators.required]),
        recaptcha: new FormControl(null, Validators.required),
      },
      { validator: CustomValidators.MatchValidator('password', 'password_confirmation') },
    );

    this.setting$.subscribe(seting => {
      if (
        (seting?.google_reCaptcha && !seting?.google_reCaptcha?.status) ||
        !seting?.google_reCaptcha
      ) {
        this.form.removeControl('recaptcha');
        this.reCaptcha = false;
      } else {
        this.form.setControl('recaptcha', new FormControl(null, Validators.required));
        this.reCaptcha = true;
      }
    });
  }

  get passwordMatchError() {
    return this.form.getError('mismatch') && this.form.get('password_confirmation')?.touched;
  }

  submit() {
    this.form.markAllAsTouched();
    if (this.tnc.invalid) {
      return;
    }
    if (this.form.valid) {
      this.store.dispatch(new RegisterAction(this.form.value)).subscribe({
        complete: () => {
          void this.router.navigateByUrl('/account/dashboard');
        },
      });
    }
  }
}
