import { CommonModule } from '@angular/common';
import { Component, inject } from '@angular/core';
import { ActivatedRoute, RouterModule } from '@angular/router';

import { TranslateModule } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Observable } from 'rxjs';

import { BlogSidebar } from './sidebar/sidebar';
import { SkeletonBlog } from './skeleton-blog/skeleton-blog';
import { GetBlogsAction } from '../../shared/action/blog.action';
import { Breadcrumb } from '../../shared/components/widgets/breadcrumb/breadcrumb';
import { NoData } from '../../shared/components/widgets/no-data/no-data';
import { Pagination } from '../../shared/components/widgets/pagination/pagination';
import { IBlogModel } from '../../shared/interface/blog.interface';
import { IBreadcrumb } from '../../shared/interface/breadcrumb';
import { IOption } from '../../shared/interface/theme-option.interface';
import { SummaryPipe } from '../../shared/pipe/summary.pipe';
import { BlogService } from '../../shared/services/blog.service';
import { BlogState } from '../../shared/state/blog.state';
import { ThemeOptionState } from '../../shared/state/theme-option.state';

@Component({
  selector: 'app-blog',
  imports: [
    TranslateModule,
    RouterModule,
    CommonModule,
    SummaryPipe,
    Breadcrumb,
    SkeletonBlog,
    Pagination,
    NoData,
    BlogSidebar,
  ],
  templateUrl: './blog.html',
  styleUrl: './blog.scss',
})
export class Blog {
  private route = inject(ActivatedRoute);
  private store = inject(Store);
  public blogService = inject(BlogService);

  blog$: Observable<IBlogModel> = inject(Store).select(BlogState.blog) as Observable<IBlogModel>;
  themeOption$: Observable<IOption> = inject(Store).select(
    ThemeOptionState.themeOptions,
  ) as Observable<IOption>;

  public breadcrumb: IBreadcrumb = {
    title: 'Blogs',
    items: [],
  };

  public filter = {
    page: 1, // Current page number
    paginate: 12, // Display per page,
    status: 1,
    category: '',
    tag: '',
  };

  public totalItems: number = 0;
  public skeletonItems = Array.from({ length: 9 }, (_, index) => index);

  public style: string;
  public sidebar: string = 'left_sidebar';

  constructor() {
    this.route.queryParams.subscribe(params => {
      this.filter.category = params['category'] ? params['category'] : '';
      this.filter.tag = params['tag'] ? params['tag'] : '';

      this.breadcrumb.items = [];
      this.breadcrumb.title = this.filter.category
        ? `Blogs: ${this.filter.category.replaceAll('-', ' ')}`
        : this.filter.tag
          ? `Blogs: ${this.filter.tag.replaceAll('-', ' ')}`
          : 'Blogs';
      this.breadcrumb.items.push({ label: 'Blogs', active: true });

      this.store.dispatch(new GetBlogsAction(this.filter));

      // For Demo Purpose only
      if (params['style']) {
        this.style = params['style'];
      }

      if (params['sidebar']) {
        this.sidebar = params['sidebar'];
      }

      if (!params['style'] && !params['sidebar']) {
        // Get Blog Layout
        this.themeOption$.subscribe(theme => {
          this.style = theme?.blog?.blog_style;
          this.sidebar = theme?.blog.blog_sidebar_type;
        });
      }
    });
    this.blog$.subscribe(blog => (this.totalItems = blog?.total));
  }

  setPaginate(data: number) {
    this.filter.page = data;
    this.store.dispatch(new GetBlogsAction(this.filter));
  }
}
