import { Component, inject } from '@angular/core';
import { ActivatedRoute, Params } from '@angular/router';

import { Store } from '@ngxs/store';
import { Observable, Subscription } from 'rxjs';

import { GetProductsAction } from '../../../shared/action/product.action';
import { Breadcrumb } from '../../../shared/components/widgets/breadcrumb/breadcrumb';
import { IBreadcrumb } from '../../../shared/interface/breadcrumb';
import { ICategory } from '../../../shared/interface/category.interface';
import { IProductModel } from '../../../shared/interface/product.interface';
import { CategoryState } from '../../../shared/state/category.state';
import { ProductState } from '../../../shared/state/product.state';
import { CollectionProducts } from '../collection/widgets/collection-products/collection-products';
import { Sidebar } from '../collection/widgets/sidebar/sidebar';

@Component({
  selector: 'app-category',
  imports: [Breadcrumb, Sidebar, CollectionProducts],
  templateUrl: './category.html',
  styleUrl: './category.scss',
})
export class Category {
  private store = inject(Store);
  private route = inject(ActivatedRoute);

  product$: Observable<IProductModel> = inject(Store).select(ProductState.product);
  category$: Observable<ICategory> = inject(Store).select(
    CategoryState.selectedCategory,
  ) as Observable<ICategory>;

  public breadcrumb: IBreadcrumb = {
    title: 'Category',
    items: [{ label: '', active: false }],
  };
  public layout: string = 'collection_category_slider';
  public skeleton: boolean = true;
  public activeCategory: string | null;
  public category: ICategory;
  public filter: Params = {
    page: 1, // Current page number
    paginate: 40, // Display per page,
    status: 1,
    field: 'created_at',
    price: '',
    category: '',
    tag: '',
    sort: 'asc', // ASC, DSC
    sortBy: 'asc',
    rating: '',
    attribute: '',
  };

  private subscriptions: Subscription = new Subscription();
  public totalItems: number = 0;

  constructor() {
    if (this.route.snapshot.paramMap.get('slug')) {
      this.activeCategory = this.route.snapshot.paramMap.get('slug');
      this.filter['category'] = this.activeCategory;
    }
  }

  ngOnInit() {
    this.subscriptions.add(
      this.category$.subscribe(category => {
        this.category = category;
        this.updateBreadcrumb();
        this.updateFilterAndFetchProducts();
        0;
      }),
    );

    this.filter['category'] = this.route.snapshot.paramMap.get('slug');
    this.store.dispatch(new GetProductsAction(this.filter));
  }

  private updateBreadcrumb() {
    this.breadcrumb.title = `Category: ${this.category?.name}`;
    this.breadcrumb.items[0].label = this.category?.name;
  }

  private updateFilterAndFetchProducts() {
    if (this.category) {
      this.filter['category'] = this.category.slug;
    }
    this.store.dispatch(new GetProductsAction(this.filter));
  }

  public changePage(page: number) {
    this.filter['category'] = page;
    this.updateFilterAndFetchProducts();
  }

  public changePaginate(paginate: number) {
    this.filter['paginate'] = paginate;
    this.updateFilterAndFetchProducts();
  }
}
