import { CommonModule } from '@angular/common';
import { Component, inject, Input, SimpleChanges, input } from '@angular/core';
import { Router, RouterModule } from '@angular/router';

import { NgbModule } from '@ng-bootstrap/ng-bootstrap';
import { TranslateModule } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Observable } from 'rxjs';

import { AddToCartAction } from '../../../../../../shared/action/cart.action';
import { AddToCompareAction } from '../../../../../../shared/action/compare.action';
import {
  AddToWishlistAction,
  DeleteWishlistAction,
} from '../../../../../../shared/action/wishlist.action';
import { Button } from '../../../../../../shared/components/widgets/button/button';
import { SaleTimer } from '../../../../../../shared/components/widgets/sale-timer/sale-timer';
import { VariantAttributes } from '../../../../../../shared/components/widgets/variant-attributes/variant-attributes';
import { ICart, ICartAddOrUpdate } from '../../../../../../shared/interface/cart.interface';
import { IProduct, IVariation } from '../../../../../../shared/interface/product.interface';
import { IOption } from '../../../../../../shared/interface/theme-option.interface';
import { CurrencySymbolPipe } from '../../../../../../shared/pipe/currency-symbol.pipe';
import { CartState } from '../../../../../../shared/state/cart.state';
import { WishlistState } from '../../../../../../shared/state/wishlist.state';
import { ProductWholesale } from '../product-wholesale/product-wholesale';

@Component({
  selector: 'app-product-contain',
  imports: [
    CommonModule,
    TranslateModule,
    RouterModule,
    NgbModule,
    CurrencySymbolPipe,
    VariantAttributes,
    SaleTimer,
    Button,
    ProductWholesale,
  ],
  templateUrl: './product-contain.html',
  styleUrl: './product-contain.scss',
})
export class ProductContain {
  private store = inject(Store);
  private router = inject(Router);

  @Input() product: IProduct;
  @Input() option: IOption | null;
  readonly owlCar = input<any>();

  cartItem$: Observable<ICart[]> = inject(Store).select(CartState.cartItems);
  wishlistIds$: Observable<number[]> = inject(Store).select(WishlistState.wishlistIds);

  public cartItem: ICart | null;
  public productQty: number = 1;
  public selectedVariation: IVariation | null;
  public totalPrice: number = 0;

  public ordersCount: number = 10;
  public viewsCount: number = 30;
  public countsInterval: ReturnType<typeof setInterval>;

  ngOnChanges(changes: SimpleChanges) {
    if (changes['product'] && changes['product'].currentValue) {
      this.selectedVariation = null;
      clearInterval(this.countsInterval);
      this.product = changes['product']?.currentValue;
    }

    this.countsInterval = setInterval(() => {
      let encourage_max_view_count = this.option?.product?.encourage_max_view_count
        ? this.option?.product?.encourage_max_view_count
        : 100;
      this.viewsCount = Math.floor(Math.random() * encourage_max_view_count) + 1;
    }, 5000);

    this.countsInterval = setInterval(() => {
      let encourage_max_order_count = this.option?.product?.encourage_max_order_count
        ? this.option?.product?.encourage_max_order_count
        : 100;
      this.ordersCount = Math.floor(Math.random() * encourage_max_order_count) + 1;
    }, 60000);

    this.cartItem$.subscribe(items => {
      this.cartItem = items.find(item => {
        if (item.variation_id) {
          this.product.variations.find(i => {
            return i.id == item.variation_id;
          });
        } else {
          return item?.product?.id == this.product.id;
        }
      })!;
    });
  }

  ngOnInit() {
    this.wholesalePriceCal();
  }

  selectVariation(variation: IVariation) {
    this.selectedVariation = variation;
  }

  updateQuantity(qty: number) {
    if (1 > this.productQty + qty) return;
    this.productQty = this.productQty + qty;
    this.wholesalePriceCal();
  }

  openDocument(url: string): void {
    if (url) {
      window.open(url, '_blank');
    } else {
      console.error('Document URL is not available');
    }
  }

  addToCart(product: IProduct, buyNow?: boolean) {
    if (product) {
      const params: ICartAddOrUpdate = {
        id:
          this.cartItem &&
          this.selectedVariation &&
          this.cartItem?.variation &&
          this.selectedVariation?.id == this.cartItem?.variation?.id
            ? this.cartItem.id
            : null,
        product_id: product?.id,
        product: product ? product : null,
        variation: this.selectedVariation ? this.selectedVariation : null,
        variation_id: this.selectedVariation?.id ? this.selectedVariation?.id : null,
        quantity: this.productQty,
      };
      this.store.dispatch(new AddToCartAction(params)).subscribe({
        complete: () => {
          if (buyNow) {
            void this.router.navigate(['/checkout']);
          }
        },
      });
    }
  }

  addToWishlist(product: IProduct) {
    product['is_wishlist'] = !product['is_wishlist'];
    let action = product['is_wishlist']
      ? new AddToWishlistAction({ product_id: product.id })
      : new DeleteWishlistAction(product.id);
    if (action) {
      this.store.dispatch(action);
    }
  }

  addToCompare(id: number) {
    this.store.dispatch(new AddToCompareAction({ product_id: id }));
  }

  externalProductLink(link: string) {
    if (link) {
      window.open(link, '_blank');
    }
  }

  wholesalePriceCal() {
    let wholesale =
      this.product?.wholesales.find(
        value => value.min_qty <= this.productQty && value.max_qty >= this.productQty,
      ) || null;
    if (wholesale && this.product?.wholesale_price_type == 'fixed') {
      this.totalPrice = this.productQty * wholesale.value;
    } else if (wholesale && this.product?.wholesale_price_type == 'percentage') {
      this.totalPrice =
        this.productQty *
        (this.selectedVariation ? this.selectedVariation.sale_price : this.product?.sale_price);
      this.totalPrice = this.totalPrice - this.totalPrice * (wholesale.value / 100);
    } else {
      this.totalPrice =
        this.productQty *
        (this.selectedVariation ? this.selectedVariation.sale_price : this.product?.sale_price);
    }
  }

  ngOnDestroy() {
    // Clear the interval when the component is destroyed
    if (this.countsInterval) {
      clearInterval(this.countsInterval);
    }
  }
}
