import { CommonModule } from '@angular/common';
import { Component, HostListener, inject } from '@angular/core';
import { ActivatedRoute } from '@angular/router';

import { Store } from '@ngxs/store';
import { Observable } from 'rxjs';

import { ProductAccordion } from './product-details/product-accordion/product-accordion';
import { ProductDigital } from './product-details/product-digital/product-digital';
import { ProductImages } from './product-details/product-images/product-images';
import { ProductSlider } from './product-details/product-slider/product-slider';
import { ProductSticky } from './product-details/product-sticky/product-sticky';
import { ProductThumbnail } from './product-details/product-thumbnail/product-thumbnail';
import { RelatedProducts } from './product-details/widgets/related-products/related-products';
import { StickyCheckout } from './product-details/widgets/sticky-checkout/sticky-checkout';
import { Breadcrumb } from '../../../shared/components/widgets/breadcrumb/breadcrumb';
import { BreadcrumbDigital } from '../../../shared/components/widgets/breadcrumb-digital/breadcrumb-digital';
import { IBreadcrumb } from '../../../shared/interface/breadcrumb';
import { IProduct } from '../../../shared/interface/product.interface';
import { IOption } from '../../../shared/interface/theme-option.interface';
import { ProductState } from '../../../shared/state/product.state';
import { ThemeOptionState } from '../../../shared/state/theme-option.state';

@Component({
  selector: 'app-product',
  imports: [
    CommonModule,
    Breadcrumb,
    BreadcrumbDigital,
    ProductThumbnail,
    RelatedProducts,
    StickyCheckout,
    ProductImages,
    ProductSlider,
    ProductSticky,
    ProductAccordion,
    ProductDigital,
  ],
  templateUrl: './product.html',
  styleUrl: './product.scss',
})
export class Product {
  private route = inject(ActivatedRoute);

  product$: Observable<IProduct> = inject(Store).select(
    ProductState.selectedProduct,
  ) as Observable<IProduct>;
  themeOptions$: Observable<IOption> = inject(Store).select(
    ThemeOptionState.themeOptions,
  ) as Observable<IOption>;

  public breadcrumb: IBreadcrumb = {
    title: 'Product',
    items: [],
  };
  public layout: string = 'product_digital';
  public product: IProduct;
  public isScrollActive = false;

  constructor() {
    this.product$.subscribe(product => {
      this.breadcrumb.items = [];
      this.breadcrumb.title = product.name;
      this.breadcrumb.items.push(
        { label: 'Product', active: true },
        { label: product.name, active: false },
      );
    });

    // For Demo Purpose only
    this.route.queryParams.subscribe(params => {
      if (params['layout']) {
        this.layout = params['layout'];
      } else {
        // Get Product Layout
        this.themeOptions$.subscribe(option => {
          this.layout =
            option?.product && option?.product?.product_layout
              ? option?.product?.product_layout
              : 'product_thumbnail';
        });
      }
    });
  }

  @HostListener('window:scroll', ['$event'])
  onScroll() {
    const button = document.querySelector('.scroll-button');
    if (button) {
      const buttonRect = button.getBoundingClientRect();
      if (buttonRect.bottom < window.innerHeight && buttonRect.bottom < 0) {
        this.isScrollActive = true;
        document.body.classList.add('stickyCart');
      } else {
        this.isScrollActive = false;
        document.body.classList.remove('stickyCart');
      }
    }
  }

  ngOnDestroy() {
    document.body.classList.remove('stickyCart');
  }
}
