import { CommonModule } from '@angular/common';
import { Component, inject } from '@angular/core';
import { ActivatedRoute } from '@angular/router';

import { Store } from '@ngxs/store';
import { Observable } from 'rxjs';

import { SellerStoreBasic } from './seller-store-basic/seller-store-basic';
import { SellerStoreClassic } from './seller-store-classic/seller-store-classic';
import { GetStoreProductsAction } from '../../../../shared/action/product.action';
import { GetStoresAction } from '../../../../shared/action/store.action';
import { Breadcrumb } from '../../../../shared/components/widgets/breadcrumb/breadcrumb';
import { Pagination } from '../../../../shared/components/widgets/pagination/pagination';
import { IBreadcrumb } from '../../../../shared/interface/breadcrumb';
import { IStoresModel } from '../../../../shared/interface/store.interface';
import { IOption } from '../../../../shared/interface/theme-option.interface';
import { StoreService } from '../../../../shared/services/store.service';
import { StoreState } from '../../../../shared/state/store.state';
import { ThemeOptionState } from '../../../../shared/state/theme-option.state';

@Component({
  selector: 'app-seller-store',
  imports: [CommonModule, Breadcrumb, SellerStoreBasic, SellerStoreClassic, Pagination],
  templateUrl: './seller-store.html',
  styleUrl: './seller-store.scss',
})
export class SellerStore {
  private route = inject(ActivatedRoute);
  public store = inject(Store);
  public storeService = inject(StoreService);

  themeOptions$: Observable<IOption> = inject(Store).select(
    ThemeOptionState.themeOptions,
  ) as Observable<IOption>;
  store$: Observable<IStoresModel> = inject(Store).select(StoreState.store);

  public breadcrumb: IBreadcrumb = {
    title: 'Seller Stores',
    items: [{ label: 'Seller Stores', active: true }],
  };
  public totalItems: number = 0;
  public filter = {
    status: 1,
    page: 1, // Current page number
    paginate: 9, // Display per page,
  };

  public skeletonItems = Array.from({ length: 6 }, (_, index) => index);
  public layout: string = 'basic_store';

  constructor() {
    // Params For Demo Purpose only
    this.route.queryParams.subscribe(params => {
      this.store.dispatch(new GetStoresAction(this.filter));
      this.store$.subscribe(store => (this.totalItems = store?.total));

      if (params['layout']) {
        this.layout = params['layout'];
      } else {
        this.themeOptions$.subscribe(option => {
          this.layout =
            option.seller && option.seller.store_layout
              ? option.seller.store_layout
              : 'basic_store';
        });
      }
    });

    this.store$.subscribe(store => {
      const storeIds = store?.data.map(store => store.id);
      if (Array.isArray(storeIds) && storeIds.length) {
        this.store.dispatch(
          new GetStoreProductsAction({ status: 1, store_ids: storeIds?.join(',') }),
        );
      }
    });
  }

  setPaginate(data: number) {
    this.filter.page = data;
    this.store.dispatch(new GetStoresAction(this.filter));
  }
}
