import { Component, inject, Input, input } from '@angular/core';
import { ActivatedRoute } from '@angular/router';

import { Store } from '@ngxs/store';
import { forkJoin, Observable } from 'rxjs';

import { GetBrandsAction } from '../../../shared/action/brand.action';
import { GetProductByIdsAction } from '../../../shared/action/product.action';
import { GetStoresAction } from '../../../shared/action/store.action';
import { ImageLink } from '../../../shared/components/widgets/image-link/image-link';
import { Title } from '../../../shared/components/widgets/title/title';
import * as data from '../../../shared/data/owl-carousel';
import { IOption } from '../../../shared/interface/theme-option.interface';
import { IBerlin } from '../../../shared/interface/theme.interface';
import { ThemeOptionService } from '../../../shared/services/theme-option.service';
import { ThemeOptionState } from '../../../shared/state/theme-option.state';
import { Brand } from '../widgets/brand/brand';
import { ThemeCategories } from '../widgets/categories/categories';
import { HomeBanner } from '../widgets/home-banner/home-banner';
import { INewsletter } from '../widgets/newsletter/newsletter';
import { Product } from '../widgets/product/product';
import { Service } from '../widgets/service/service';
import { TopSeller } from '../widgets/top-seller/top-seller';

@Component({
  selector: 'app-berlin',
  imports: [
    HomeBanner,
    Service,
    Title,
    Product,
    ThemeCategories,
    ImageLink,
    TopSeller,
    Brand,
    INewsletter,
  ],
  templateUrl: './berlin.html',
  styleUrl: './berlin.scss',
})
export class Berlin {
  private themeOptionService = inject(ThemeOptionService);
  private route = inject(ActivatedRoute);
  private store = inject(Store);

  @Input() data?: IBerlin;
  readonly slug = input<string>();

  themeOption$: Observable<IOption> = inject(Store).select(
    ThemeOptionState.themeOptions,
  ) as Observable<IOption>;

  public categorySlider = data.categorySlider;
  public productSliderMargin = data.productSliderMargin;

  ngOnInit() {
    if (this.data?.slug == this.slug()) {
      const getProducts$ = this.store.dispatch(
        new GetProductByIdsAction({
          status: 1,
          paginate: this.data?.content?.products_ids.length,
          ids: this.data?.content?.products_ids?.join(','),
        }),
      );
      const getBrand$ = this.store.dispatch(
        new GetBrandsAction({
          status: 1,
          ids: this.data?.content?.brands?.brand_ids?.join(),
        }),
      );
      const getStore$ = this.store.dispatch(
        new GetStoresAction({
          status: 1,
          ids: this.data?.content?.main_content?.seller?.store_ids?.join(),
        }),
      );

      // Skeleton Loader
      document.body.classList.add('skeleton-body');

      forkJoin([getProducts$, getBrand$, getStore$]).subscribe({
        complete: () => {
          document.body.classList.remove('skeleton-body');
          this.themeOptionService.preloader = false;
        },
      });

      this.route.queryParams.subscribe(_params => {
        if (this.route.snapshot.data['data'].theme_option.productBox === 'digital') {
          if (
            this.productSliderMargin &&
            this.productSliderMargin.responsive &&
            this.productSliderMargin.responsive['1030']
          ) {
            this.productSliderMargin = {
              ...this.productSliderMargin,
              items: 3,
              responsive: {
                ...this.productSliderMargin.responsive,
                1030: {
                  items: 3,
                },
              },
            };
          }
        } else {
          if (
            this.productSliderMargin &&
            this.productSliderMargin.responsive &&
            this.productSliderMargin.responsive['1030']
          ) {
            this.productSliderMargin = {
              ...this.productSliderMargin,
              items: 5,
              responsive: {
                ...this.productSliderMargin.responsive,
                1030: {
                  items: 5,
                },
              },
            };
          }
        }
      });
    }
  }
}
