import { CommonModule } from '@angular/common';
import { Component, inject, Input, input } from '@angular/core';
import { ActivatedRoute } from '@angular/router';

import { TranslateModule } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { forkJoin, Observable } from 'rxjs';

import { environment } from '../../../../environments/environment';
import { GetBlogsAction } from '../../../shared/action/blog.action';
import { GetBrandsAction } from '../../../shared/action/brand.action';
import {
  GetCategoryProductsAction,
  GetProductByIdsAction,
} from '../../../shared/action/product.action';
import { GetStoresAction } from '../../../shared/action/store.action';
import { Search } from '../../../shared/components/header/widgets/search/search';
import { Title } from '../../../shared/components/widgets/title/title';
import * as data from '../../../shared/data/owl-carousel';
import { ICategory } from '../../../shared/interface/category.interface';
import { IOption } from '../../../shared/interface/theme-option.interface';
import { ThemeOptionService } from '../../../shared/services/theme-option.service';
import { ThemeOptionState } from '../../../shared/state/theme-option.state';
import { Blog } from '../widgets/blog/blog';
import { Brand } from '../widgets/brand/brand';
import { ThemeCategories } from '../widgets/categories/categories';
import { CategoryProductFilter } from '../widgets/category-product-filter/category-product-filter';
import { Product } from '../widgets/product/product';
import { TopSeller } from '../widgets/top-seller/top-seller';

@Component({
  selector: 'app-cairo',
  imports: [
    CommonModule,
    TranslateModule,
    Search,
    ThemeCategories,
    Title,
    Product,
    TopSeller,
    CategoryProductFilter,
    Blog,
    Brand,
  ],
  templateUrl: './cairo.html',
  styleUrl: './cairo.scss',
})
export class Cairo {
  private themeOptionService = inject(ThemeOptionService);
  private route = inject(ActivatedRoute);
  private store = inject(Store);

  @Input() data?: any;
  readonly slug = input<string>();

  public skeletonItems = Array.from({ length: 6 }, (_, index) => index);
  public storageURL = environment.storageURL;

  themeOption$: Observable<IOption> = inject(Store).select(
    ThemeOptionState.themeOptions,
  ) as Observable<IOption>;

  public categorySlider = data.categorySlider9;
  public productSlider = data.customOptionsItem3;
  public bannerSlider = data.bannerSlider;

  public showProductBox = 6;
  public productFilterIds: number[] = [];
  public categories: ICategory[] = [];
  public path: string;

  ngOnInit() {
    this.themeOptionService.isDigitalProductBox = true;
    if (this.data?.slug == this.slug()) {
      const getProducts$ = this.store.dispatch(
        new GetProductByIdsAction({
          status: 1,
          paginate: this.data?.content?.products_ids.length,
          ids: this.data?.content?.products_ids?.join(','),
        }),
      );
      const getBrand$ = this.store.dispatch(
        new GetBrandsAction({
          status: 1,
          ids: this.data?.content?.brands?.brand_ids?.join(),
        }),
      );
      const getStore$ = this.store.dispatch(
        new GetStoresAction({
          status: 1,
          ids: this.data?.content?.seller?.store_ids?.join(),
        }),
      );
      const getBlogs$ = this.store.dispatch(
        new GetBlogsAction({
          status: 1,
        }),
      );
      const getCategoryProduct$ = this.store.dispatch(
        new GetCategoryProductsAction({
          category_ids: this.data.content?.categories_products?.category_ids?.join(','),
        }),
      );

      // Skeleton Loader
      document.body.classList.add('skeleton-body');
      document.body.classList.add('cairo');

      forkJoin([getProducts$, getBrand$, getStore$, getBlogs$, getCategoryProduct$]).subscribe({
        complete: () => {
          document.body.classList.remove('skeleton-body');
          this.themeOptionService.preloader = false;
        },
      });
    }

    this.route.queryParams.subscribe(_params => {
      if (this.route.snapshot.data['data'].theme_option.productBox === 'digital') {
        if (
          this.productSlider &&
          this.productSlider.responsive &&
          this.productSlider.responsive['1050']
        ) {
          this.productSlider = {
            ...this.productSlider,
            items: 3,
            responsive: {
              ...this.productSlider.responsive,
              1050: {
                items: 3,
              },
            },
          };
          this.showProductBox = 4;
        }
      } else {
        if (
          this.productSlider &&
          this.productSlider.responsive &&
          this.productSlider.responsive['1050']
        ) {
          this.productSlider = {
            ...this.productSlider,
            items: 4,
            responsive: {
              ...this.productSlider.responsive,
              1050: {
                items: 4,
              },
            },
          };
          this.showProductBox = 6;
        }
      }
    });
  }

  ngOnDestroy() {
    document.body.classList.remove('cairo');
  }
}
