import { Component, inject, Input, input } from '@angular/core';
import { ActivatedRoute } from '@angular/router';

import { Store } from '@ngxs/store';
import { forkJoin, Observable } from 'rxjs';

import { GetBlogsAction } from '../../../shared/action/blog.action';
import { GetBrandsAction } from '../../../shared/action/brand.action';
import { GetProductByIdsAction } from '../../../shared/action/product.action';
import { GetStoresAction } from '../../../shared/action/store.action';
import { Title } from '../../../shared/components/widgets/title/title';
import * as data from '../../../shared/data/owl-carousel';
import { IOption } from '../../../shared/interface/theme-option.interface';
import { IOsaka } from '../../../shared/interface/theme.interface';
import { ThemeOptionService } from '../../../shared/services/theme-option.service';
import { ThemeOptionState } from '../../../shared/state/theme-option.state';
import { Banner } from '../widgets/banner/banner';
import { Blog } from '../widgets/blog/blog';
import { Brand } from '../widgets/brand/brand';
import { ThemeCategories } from '../widgets/categories/categories';
import { Collection } from '../widgets/collection/collection';
import { FourColumnProduct } from '../widgets/four-column-product/four-column-product';
import { HomeBanner } from '../widgets/home-banner/home-banner';
import { INewsletter } from '../widgets/newsletter/newsletter';
import { Product } from '../widgets/product/product';
import { TopSeller } from '../widgets/top-seller/top-seller';

@Component({
  selector: 'app-osaka',
  imports: [
    HomeBanner,
    Title,
    ThemeCategories,
    Banner,
    Product,
    TopSeller,
    Collection,
    FourColumnProduct,
    Blog,
    Brand,
    INewsletter,
  ],
  templateUrl: './osaka.html',
  styleUrl: './osaka.scss',
})
export class Osaka {
  private themeOptionService = inject(ThemeOptionService);
  private route = inject(ActivatedRoute);
  private store = inject(Store);

  @Input() data?: IOsaka;
  readonly slug = input<string>();

  public categorySlider = data.categorySlider9;
  public productSlider6Item = data.productSlider6Item;
  public productSlider = data.bannerSlider;

  themeOption$: Observable<IOption> = inject(Store).select(
    ThemeOptionState.themeOptions,
  ) as Observable<IOption>;

  ngOnInit() {
    if (this.data?.slug == this.slug()) {
      const getProducts$ = this.store.dispatch(
        new GetProductByIdsAction({
          status: 1,
          paginate: this.data?.content?.products_ids.length,
          ids: this.data?.content?.products_ids?.join(','),
        }),
      );
      const getBrand$ = this.store.dispatch(
        new GetBrandsAction({
          status: 1,
          ids: this.data?.content?.brands?.brand_ids?.join(),
        }),
      );
      const getStore$ = this.store.dispatch(
        new GetStoresAction({
          status: 1,
          ids: this.data?.content?.seller?.store_ids?.join(),
        }),
      );
      const getBlogs$ = this.store.dispatch(
        new GetBlogsAction({
          status: 1,
          ids: this.data?.content?.featured_blogs?.blog_ids?.join(','),
        }),
      );

      // Skeleton Loader
      document.body.classList.add('skeleton-body');

      forkJoin([getProducts$, getBrand$, getStore$, getBlogs$]).subscribe({
        complete: () => {
          document.body.classList.remove('skeleton-body');
          this.themeOptionService.preloader = false;
        },
      });
    }

    this.route.queryParams.subscribe(_params => {
      if (this.route.snapshot.data['data'].theme_option.productBox === 'digital') {
        if (
          this.productSlider6Item &&
          this.productSlider6Item.responsive &&
          this.productSlider6Item.responsive['1065']
        ) {
          this.productSlider6Item = {
            ...this.productSlider6Item,
            items: 4,
            responsive: {
              ...this.productSlider6Item.responsive,
              1065: {
                items: 4,
              },
            },
          };
        }
      } else {
        if (
          this.productSlider6Item &&
          this.productSlider6Item.responsive &&
          this.productSlider6Item.responsive['1065']
        ) {
          this.productSlider6Item = {
            ...this.productSlider6Item,
            items: 6,
            responsive: {
              ...this.productSlider6Item.responsive,
              1065: {
                items: 6,
              },
            },
          };
        }
      }
    });
  }
}
