import { CommonModule } from '@angular/common';
import { Component, inject, Input, input } from '@angular/core';
import { ActivatedRoute } from '@angular/router';

import { Store } from '@ngxs/store';
import { forkJoin, Observable } from 'rxjs';

import { GetBlogsAction } from '../../../shared/action/blog.action';
import { GetBrandsAction } from '../../../shared/action/brand.action';
import { GetProductByIdsAction } from '../../../shared/action/product.action';
import { GetStoresAction } from '../../../shared/action/store.action';
import { ImageLink } from '../../../shared/components/widgets/image-link/image-link';
import { Title } from '../../../shared/components/widgets/title/title';
import * as data from '../../../shared/data/owl-carousel';
import { IOption } from '../../../shared/interface/theme-option.interface';
import { IParis } from '../../../shared/interface/theme.interface';
import { ThemeOptionService } from '../../../shared/services/theme-option.service';
import { ThemeOptionState } from '../../../shared/state/theme-option.state';
import { Banner } from '../widgets/banner/banner';
import { Blog } from '../widgets/blog/blog';
import { Brand } from '../widgets/brand/brand';
import { ThemeCategories } from '../widgets/categories/categories';
import { HomeBanner } from '../widgets/home-banner/home-banner';
import { INewsletter } from '../widgets/newsletter/newsletter';
import { Product } from '../widgets/product/product';
import { TopSeller } from '../widgets/top-seller/top-seller';

@Component({
  selector: 'app-paris',
  imports: [
    HomeBanner,
    Banner,
    ThemeCategories,
    Product,
    Title,
    ImageLink,
    TopSeller,
    Blog,
    Brand,
    INewsletter,
    CommonModule,
  ],
  templateUrl: './paris.html',
  styleUrl: './paris.scss',
})
export class Paris {
  private themeOptionService = inject(ThemeOptionService);
  private route = inject(ActivatedRoute);
  private store = inject(Store);

  @Input() data?: IParis;
  readonly slug = input<string>();

  themeOption$: Observable<IOption> = inject(Store).select(
    ThemeOptionState.themeOptions,
  ) as Observable<IOption>;

  public categorySlider = data.categorySlider;
  public productSlider = data.productSlider;

  ngOnInit() {
    if (this.data?.slug == this.slug()) {
      const getProducts$ = this.store.dispatch(
        new GetProductByIdsAction({
          status: 1,
          paginate: this.data?.content?.products_ids.length,
          ids: this.data?.content?.products_ids?.join(','),
        }),
      );
      const getBrand$ = this.store.dispatch(
        new GetBrandsAction({
          status: 1,
          ids: this.data?.content?.brands?.brand_ids?.join(),
        }),
      );
      const getStore$ = this.store.dispatch(
        new GetStoresAction({
          status: 1,
          ids: this.data?.content?.main_content?.seller?.store_ids?.join(),
        }),
      );
      const getBlogs$ = this.store.dispatch(
        new GetBlogsAction({
          status: 1,
          ids: this.data?.content.main_content?.section9_featured_blogs?.blog_ids?.join(','),
        }),
      );

      // Skeleton Loader
      document.body.classList.add('skeleton-body');

      forkJoin([getProducts$, getBlogs$, getBrand$, getStore$]).subscribe({
        complete: () => {
          document.body.classList.remove('skeleton-body');
          this.themeOptionService.preloader = false;
        },
      });
    }
    this.route.queryParams.subscribe(_params => {
      if (this.route.snapshot.data['data'].theme_option.productBox === 'digital') {
        if (
          this.productSlider &&
          this.productSlider.responsive &&
          this.productSlider.responsive['1000']
        ) {
          this.productSlider = {
            ...this.productSlider,
            items: 3,
            responsive: {
              ...this.productSlider.responsive,
              1000: {
                items: 3,
              },
            },
          };
        }
      } else {
        if (
          this.productSlider &&
          this.productSlider.responsive &&
          this.productSlider.responsive['1000']
        ) {
          this.productSlider = {
            ...this.productSlider,
            items: 5,
            responsive: {
              ...this.productSlider.responsive,
              1000: {
                items: 5,
              },
            },
          };
        }
      }
    });
  }
}
