import { Component, inject, Input, input } from '@angular/core';
import { ActivatedRoute } from '@angular/router';

import { Store } from '@ngxs/store';
import { forkJoin, Observable } from 'rxjs';

import { environment } from '../../../../environments/environment';
import { GetBrandsAction } from '../../../shared/action/brand.action';
import { GetProductByIdsAction } from '../../../shared/action/product.action';
import { GetStoresAction } from '../../../shared/action/store.action';
import { ImageLink } from '../../../shared/components/widgets/image-link/image-link';
import { Title } from '../../../shared/components/widgets/title/title';
import * as data from '../../../shared/data/owl-carousel';
import { IOption } from '../../../shared/interface/theme-option.interface';
import { ITokyo } from '../../../shared/interface/theme.interface';
import { ThemeOptionService } from '../../../shared/services/theme-option.service';
import { ThemeOptionState } from '../../../shared/state/theme-option.state';
import { ThemeState } from '../../../shared/state/theme.state';
import { Banner } from '../widgets/banner/banner';
import { Brand } from '../widgets/brand/brand';
import { ThemeCategories } from '../widgets/categories/categories';
import { FourColumnProduct } from '../widgets/four-column-product/four-column-product';
import { HomeBanner } from '../widgets/home-banner/home-banner';
import { INewsletter } from '../widgets/newsletter/newsletter';
import { Product } from '../widgets/product/product';
import { TopSeller } from '../widgets/top-seller/top-seller';

@Component({
  selector: 'app-tokyo',
  imports: [
    HomeBanner,
    ThemeCategories,
    Banner,
    Title,
    Product,
    TopSeller,
    ImageLink,
    FourColumnProduct,
    Brand,
    INewsletter,
  ],
  templateUrl: './tokyo.html',
  styleUrl: './tokyo.scss',
})
export class Tokyo {
  private themeOptionService = inject(ThemeOptionService);
  private route = inject(ActivatedRoute);
  private store = inject(Store);

  @Input() data?: ITokyo;
  readonly slug = input<string>();

  public productSlider = data.productSlider2;
  public categorySlider = data.categorySlider9;
  public storageURL = environment.storageURL;

  themeOption$: Observable<IOption> = inject(Store).select(
    ThemeOptionState.themeOptions,
  ) as Observable<IOption>;
  activeTheme$: Observable<string> = inject(Store).select(ThemeState.activeTheme);

  ngOnInit() {
    if (this.data?.slug == this.slug()) {
      const getProducts$ = this.store.dispatch(
        new GetProductByIdsAction({
          status: 1,
          paginate: this.data?.content?.products_ids.length,
          ids: this.data?.content?.products_ids?.join(','),
        }),
      );
      const getBrand$ = this.store.dispatch(
        new GetBrandsAction({
          status: 1,
          ids: this.data?.content?.brands?.brand_ids?.join(),
        }),
      );
      const getStore$ = this.store.dispatch(
        new GetStoresAction({
          status: 1,
          ids: this.data?.content?.main_content?.seller?.store_ids?.join(),
        }),
      );
      // Skeleton Loader
      document.body.classList.add('skeleton-body');

      forkJoin([getProducts$, getBrand$, getStore$]).subscribe({
        complete: () => {
          document.body.classList.remove('skeleton-body');
          this.themeOptionService.preloader = false;
        },
      });
    }

    this.route.queryParams.subscribe(_params => {
      if (this.route.snapshot.data['data'].theme_option.productBox === 'digital') {
        if (
          this.productSlider &&
          this.productSlider.responsive &&
          this.productSlider.responsive['990']
        ) {
          this.productSlider = {
            ...this.productSlider,
            items: 3,
            responsive: {
              ...this.productSlider.responsive,
              990: {
                items: 3,
              },
            },
          };
        }
      } else {
        if (
          this.productSlider &&
          this.productSlider.responsive &&
          this.productSlider.responsive['990']
        ) {
          this.productSlider = {
            ...this.productSlider,
            items: 5,
            responsive: {
              ...this.productSlider.responsive,
              990: {
                items: 5,
              },
            },
          };
        }
      }
    });
  }

  ngAfterViewChecked() {
    // Add Topbar Dark Class
    document.querySelector('.header-top')?.classList.add('bg-dark');
  }

  ngOnDestroy() {
    // Remove Topbar Dark Class for this theme only
    document.querySelector('.header-top')?.classList.remove('bg-dark');
  }
}
