import { CommonModule } from '@angular/common';
import { Component, inject, input, viewChild } from '@angular/core';
import { RouterModule } from '@angular/router';

import { TranslateModule } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Observable } from 'rxjs';

import {
  ClearCartAction,
  DeleteCartAction,
  ToggleSidebarCartAction,
  UpdateCartAction,
} from '../../../../action/cart.action';
import { ICart, ICartAddOrUpdate } from '../../../../interface/cart.interface';
import { IValues } from '../../../../interface/setting.interface';
import { IOption } from '../../../../interface/theme-option.interface';
import { CurrencySymbolPipe } from '../../../../pipe/currency-symbol.pipe';
import { CartService } from '../../../../services/cart.service';
import { CartState } from '../../../../state/cart.state';
import { SettingState } from '../../../../state/setting.state';
import { ThemeOptionState } from '../../../../state/theme-option.state';
import { Button } from '../../../widgets/button/button';
import { VariationModal } from '../../../widgets/modal/variation-modal/variation-modal';

@Component({
  selector: 'app-header-cart',
  imports: [
    CommonModule,
    TranslateModule,
    RouterModule,
    CurrencySymbolPipe,
    Button,
    VariationModal,
  ],
  templateUrl: './cart.html',
  styleUrl: './cart.scss',
})
export class Cart {
  private store = inject(Store);
  public cartService = inject(CartService);

  cartItem$: Observable<ICart[]> = inject(Store).select(CartState.cartItems);
  cartTotal$: Observable<number> = inject(Store).select(CartState.cartTotal);
  sidebarCartOpen$: Observable<boolean> = inject(Store).select(CartState.sidebarCartOpen);
  themeOption$: Observable<IOption> = inject(Store).select(
    ThemeOptionState.themeOptions,
  ) as Observable<IOption>;
  setting$: Observable<IValues> = inject(Store).select(SettingState.setting) as Observable<IValues>;

  readonly VariationModal = viewChild<VariationModal>('variationModal');

  readonly style = input<string>('basic');

  public cartStyle: string = 'cart_sidebar';
  public shippingFreeAmt: number = 0;
  public cartTotal: number = 0;
  public shippingCal: number = 0;
  public confettiItems = Array.from({ length: 150 }, (_, index) => index);
  public confetti: number = 0;
  public loader: boolean = false;
  public cartHide: boolean = true;

  constructor() {
    this.themeOption$.subscribe(option => (this.cartStyle = option?.general?.cart_style));

    // Calculation
    this.cartTotal$.subscribe(total => {
      this.setting$.subscribe(
        setting => (this.shippingFreeAmt = setting?.general?.min_order_free_shipping),
      );
      this.cartTotal = total;
      this.shippingCal = (this.cartTotal * 100) / this.shippingFreeAmt;
      if (this.shippingCal > 100) {
        this.shippingCal = 100;
        if (this.confetti == 0) {
          this.confetti = 1;
          setTimeout(() => {
            this.confetti = 2;
          }, 4500);
        }
      } else {
        this.confetti = 0;
      }
    });

    setTimeout(() => {
      this.cartHide = false;
    }, 1000);
  }

  cartToggle(value: boolean) {
    this.store.dispatch(new ToggleSidebarCartAction(value));
  }

  updateQuantity(item: ICart, qty: number) {
    const params: ICartAddOrUpdate = {
      id: item?.id,
      product_id: item?.product?.id,
      product: item?.product ? item?.product : null,
      variation_id: item?.variation_id ? item?.variation_id : null,
      variation: item?.variation ? item?.variation : null,
      quantity: qty,
    };
    this.store.dispatch(new UpdateCartAction(params));
    this.cartService.updateQty();
  }

  delete(id: number) {
    this.store.dispatch(new DeleteCartAction(id));
  }

  clearCart() {
    this.store.dispatch(new ClearCartAction());
  }
}
