import { Component, inject, TemplateRef, viewChild } from '@angular/core';
import { FormControl, ReactiveFormsModule, Validators } from '@angular/forms';

import { ModalDismissReasons, NgbModal } from '@ng-bootstrap/ng-bootstrap';
import { TranslateModule } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Observable } from 'rxjs';

import { environment } from '../../../../../../environments/environment';
import { UpdateSessionAction } from '../../../../../shared/action/theme-option.action';
import { ThemeOptionState } from '../../../../../shared/state/theme-option.state';
import { SubscriptionAction } from '../../../../action/subscription.action';
import { IOption } from '../../../../interface/theme-option.interface';
import { Button } from '../../button/button';

@Component({
  selector: 'app-newsletter-modal',
  imports: [TranslateModule, ReactiveFormsModule, Button],
  templateUrl: './newsletter-modal.html',
  styleUrl: './newsletter-modal.scss',
})
export class NewsletterModal {
  private modalService = inject(NgbModal);
  private store = inject(Store);

  readonly NewsletterModal = viewChild<TemplateRef<string>>('newsletterModal');

  newsletter$: Observable<boolean> = inject(Store).select(ThemeOptionState.newsletter);
  themeOption$: Observable<IOption> = inject(Store).select(
    ThemeOptionState.themeOptions,
  ) as Observable<IOption>;

  public closeResult: string;
  public modalOpen: boolean = true;
  public newsletter: boolean;
  public themeOption: IOption;
  public email = new FormControl('', [Validators.email]);
  public isSubmit: boolean = false;
  public storageURL = environment.storageURL;

  constructor() {
    this.newsletter$.subscribe(res => (this.newsletter = res));
    this.themeOption$.subscribe(res => (this.themeOption = res));
  }

  ngAfterViewInit(): void {
    if (this.newsletter === true) {
      setTimeout(() => {
        void this.openModal();
      }, 3000);
      this.store.dispatch(new UpdateSessionAction('newsletter', false));
    }
  }

  async openModal() {
    this.modalOpen = true;
    this.modalService
      .open(this.NewsletterModal(), {
        ariaLabelledBy: 'profile-Modal',
        centered: true,
        windowClass: 'theme-modal modal-lg newsletter-modal',
      })
      .result.then(
        result => {
          `Result ${result}`;
        },
        reason => {
          this.closeResult = `Dismissed ${this.getDismissReason(reason)}`;
        },
      );
  }

  private getDismissReason(reason: ModalDismissReasons): string {
    if (reason === ModalDismissReasons.ESC) {
      return 'by pressing ESC';
    } else if (reason === ModalDismissReasons.BACKDROP_CLICK) {
      return 'by clicking on a backdrop';
    } else {
      return `with: ${reason}`;
    }
  }

  submit() {
    this.isSubmit = true;
    if (this.email.valid) {
      this.store.dispatch(new SubscriptionAction({ email: this.email.value! }));
      this.email.reset();
      this.isSubmit = false;
    }
  }

  ngOnDestroy() {
    if (this.modalOpen) {
      this.modalService.dismissAll();
    }
  }
}
