import { CommonModule } from '@angular/common';
import { Component, inject, Input, input, viewChild } from '@angular/core';
import { RouterModule } from '@angular/router';

import { NgbModule, NgbRatingConfig } from '@ng-bootstrap/ng-bootstrap';
import { TranslateModule } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Observable } from 'rxjs';

import { AddToCartAction } from '../../../../../shared/action/cart.action';
import { AddToCompareAction } from '../../../../../shared/action/compare.action';
import {
  AddToWishlistAction,
  DeleteWishlistAction,
} from '../../../../../shared/action/wishlist.action';
import { ICart, ICartAddOrUpdate } from '../../../../../shared/interface/cart.interface';
import { IProduct } from '../../../../../shared/interface/product.interface';
import { CartState } from '../../../../../shared/state/cart.state';
import { CurrencySymbolPipe } from '../../../../pipe/currency-symbol.pipe';
import { TitleCasePipe } from '../../../../pipe/title-case.pipe';
import { ProductDetailModal } from '../../../widgets/modal/product-detail-modal/product-detail-modal';
import { Button } from '../../button/button';
import { VariationModal } from '../../modal/variation-modal/variation-modal';
import { ProductCartButton } from '../widgets/product-cart-button/product-cart-button';
import { ProductHoverAction } from '../widgets/product-hover-action/product-hover-action';

@Component({
  selector: 'app-basic-product-box',
  imports: [
    CommonModule,
    RouterModule,
    NgbModule,
    TranslateModule,
    CurrencySymbolPipe,
    TitleCasePipe,
    Button,
    ProductHoverAction,
    ProductCartButton,
    ProductDetailModal,
    VariationModal,
  ],
  templateUrl: './basic-product-box.html',
  styleUrl: './basic-product-box.scss',
})
export class BasicProductBox {
  private store = inject(Store);
  public config = inject(NgbRatingConfig);

  @Input() product: IProduct;
  readonly class = input<string>();
  readonly close = input<boolean>();

  cartItem$: Observable<ICart[]> = inject(Store).select(CartState.cartItems) as Observable<ICart[]>;

  readonly productDetailModal = viewChild<ProductDetailModal>('productDetailModal');
  readonly VariationModal = viewChild<VariationModal>('variationModal');

  public cartItem: ICart | null;
  public currentDate: number | null;
  public saleStartDate: number | null;

  constructor() {
    this.config.max = 5;
    this.config.readonly = true;
  }

  ngOnInit() {
    this.cartItem$.subscribe(items => {
      this.cartItem = items?.find(item => item?.product?.id == this.product.id)!;
    });
  }

  addToCart(product: IProduct, qty: number) {
    const params: ICartAddOrUpdate = {
      id: this.cartItem ? this.cartItem.id : null,
      product: product,
      product_id: product?.id,
      variation_id: this.cartItem ? this.cartItem?.variation_id : null,
      variation: this.cartItem ? this.cartItem?.variation : null,
      quantity: qty,
    };
    this.store.dispatch(new AddToCartAction(params));
  }

  addToWishlist(product: IProduct) {
    product['is_wishlist'] = !product['is_wishlist'];
    let action = product['is_wishlist']
      ? new AddToWishlistAction({ product_id: product.id })
      : new DeleteWishlistAction(product.id);
    if (action) {
      this.store.dispatch(action);
    }
  }

  removeWishlist(id: number) {
    this.store.dispatch(new DeleteWishlistAction(id));
  }

  addToCompar(id: number) {
    this.store.dispatch(new AddToCompareAction({ product_id: id }));
  }

  externalProductLink(link: string) {
    if (link) {
      window.open(link, '_blank');
    }
  }
}
