import { inject, Injectable } from '@angular/core';
import { Router } from '@angular/router';

import { Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetPagesAction,
  GetPageBySlugAction,
  ContactUsAction,
  GetFaqsAction,
} from '../action/page.action';
import { IContactUsModel, IFaq, IPage } from '../interface/page.interface';
import { PageService } from '../services/page.service';

export class PageStateModel {
  page = {
    data: [] as IPage[],
    total: 0,
  };
  faq = {
    data: [] as IFaq[],
    total: 0,
  };
  selectedPage: IPage | null;
}

@State<PageStateModel>({
  name: 'page',
  defaults: {
    page: {
      data: [],
      total: 0,
    },
    faq: {
      data: [],
      total: 0,
    },
    selectedPage: null,
  },
})
@Injectable()
export class PageState {
  private router = inject(Router);
  private pageService = inject(PageService);

  @Selector()
  static page(state: PageStateModel) {
    return state.page;
  }

  @Selector()
  static faq(state: PageStateModel) {
    return state.faq;
  }

  @Selector()
  static selectedPage(state: PageStateModel) {
    return state.selectedPage;
  }

  @Action(GetPagesAction)
  getPages(ctx: StateContext<PageStateModel>, action: GetPagesAction) {
    return this.pageService.getPages(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            page: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(GetPageBySlugAction)
  getPageBySlug(ctx: StateContext<PageStateModel>, { slug }: GetPageBySlugAction) {
    this.pageService.skeletonLoader = true;
    return this.pageService.getPageBySlug(slug).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedPage: result,
          });
        },
        complete: () => {
          this.pageService.skeletonLoader = false;
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(GetFaqsAction)
  getFaqs(ctx: StateContext<PageStateModel>) {
    this.pageService.skeletonLoader = true;
    return this.pageService.getFaqs().pipe(
      tap({
        next: result => {
          ctx.patchState({
            faq: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        complete: () => {
          this.pageService.skeletonLoader = false;
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(ContactUsAction)
  contactUs(ctx: StateContext<IContactUsModel>, { payload }: ContactUsAction) {
    return this.pageService.contactUs(payload).pipe(
      tap({
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
